/** @flectra-module */

import * as spreadsheet from "@flectra/o-spreadsheet";
import { _t } from "@web/core/l10n/translation";
import { FlectraChart } from "./flectra_chart";

const { chartRegistry } = spreadsheet.registries;

const { getDefaultChartJsRuntime, chartFontColor, ChartColors, formatValue } = spreadsheet.helpers;

export class FlectraBarChart extends FlectraChart {
    constructor(definition, sheetId, getters) {
        super(definition, sheetId, getters);
        this.verticalAxisPosition = definition.verticalAxisPosition;
        this.stacked = definition.stacked;
    }

    getDefinition() {
        return {
            ...super.getDefinition(),
            verticalAxisPosition: this.verticalAxisPosition,
            stacked: this.stacked,
        };
    }
}

chartRegistry.add("flectra_bar", {
    match: (type) => type === "flectra_bar",
    createChart: (definition, sheetId, getters) => new FlectraBarChart(definition, sheetId, getters),
    getChartRuntime: createFlectraChartRuntime,
    validateChartDefinition: (validator, definition) =>
        FlectraBarChart.validateChartDefinition(validator, definition),
    transformDefinition: (definition) => FlectraBarChart.transformDefinition(definition),
    getChartDefinitionFromContextCreation: () => FlectraBarChart.getDefinitionFromContextCreation(),
    name: _t("Bar"),
});

function createFlectraChartRuntime(chart, getters) {
    const background = chart.background || "#FFFFFF";
    const { datasets, labels } = chart.dataSource.getData();
    const locale = getters.getLocale();
    const chartJsConfig = getBarConfiguration(chart, labels, locale);
    const colors = new ChartColors();
    for (const { label, data } of datasets) {
        const color = colors.next();
        const dataset = {
            label,
            data,
            borderColor: color,
            backgroundColor: color,
        };
        chartJsConfig.data.datasets.push(dataset);
    }

    return { background, chartJsConfig };
}

function getBarConfiguration(chart, labels, locale) {
    const color = chartFontColor(chart.background);
    const config = getDefaultChartJsRuntime(chart, labels, color, { locale });
    config.type = chart.type.replace("flectra_", "");
    const legend = {
        ...config.options.legend,
        display: chart.legendPosition !== "none",
        labels: { color },
    };
    legend.position = chart.legendPosition;
    config.options.plugins = config.options.plugins || {};
    config.options.plugins.legend = legend;
    config.options.layout = {
        padding: { left: 20, right: 20, top: chart.title ? 10 : 25, bottom: 10 },
    };
    config.options.scales = {
        x: {
            ticks: {
                // x axis configuration
                maxRotation: 60,
                minRotation: 15,
                padding: 5,
                labelOffset: 2,
                color,
            },
        },
        y: {
            position: chart.verticalAxisPosition,
            ticks: {
                color,
                // y axis configuration
                callback: (value) =>
                    formatValue(value, {
                        locale,
                        format: Math.abs(value) >= 1000 ? "#,##" : undefined,
                    }),
            },
            beginAtZero: true, // the origin of the y axis is always zero
        },
    };
    if (chart.stacked) {
        config.options.scales.x.stacked = true;
        config.options.scales.y.stacked = true;
    }
    return config;
}
